/**
* \file: HIDDigitizer.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: Digital iPod Out - Wayland Adapter
*
* \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
*
* \copyright (c) 2013 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include "HIDDigitizer.h"
#include <string.h>

namespace adit { namespace carplay
{

using namespace std;

// some macros to make the HID descriptors human-readable
#define HID_USAGE_PAGE(a)                  0x05, (a)
#define HID_USAGE(a)                       0x09, (a)
#define HID_COLLECTION(a)                  0xA1, (a)
#define HID_END_COLLECTION()               0xC0
#define HID_REPORT_SIZE(size)              0x75, (size)
#define HID_REPORT_COUNT(count)            0x95, (count)
#define HID_INPUT(a)                       0x81, (a)
#define HID_LOGICAL_MINMAX(min, max)       0x15, (min), 0x25, (max)
#define HID_LOGICAL_MINMAX16(min, max)     0x16, ((min) & 0xff), (((min) >> 8) & 0xff), \
    0x26, ((max) & 0xff), (((max) >> 8) & 0xff)
/* currently unused
#define HID_USAGE_MINMAX(min, max)         0x19, (min), 0x29, (max)
#define HID_PHYSICAL_MINMAX16(min, max)    0x36, ((min) & 0xff), (((min) >> 8) & 0xff), \
    0x46, ((max) & 0xff), (((max) >> 8) & 0xff)
#define HID_UNIT(a)                        0x65, (a)
#define HID_EXPONENT(a)                    0x55, (a)
*/

// single touch digitizer descriptor
static const uint8_t _singleTouchReportLen = 5;
static const uint8_t _singleTouch[] =
{
    HID_USAGE_PAGE (0x0D),
    HID_USAGE (0x04),
    HID_COLLECTION (0x01),
        // Digitizer
        HID_USAGE_PAGE (0x0D),
        HID_USAGE (0x22),
        HID_COLLECTION (0x02),
            // Digitizer
            HID_USAGE_PAGE (0x0D),
            HID_USAGE (0x33),

            HID_LOGICAL_MINMAX(0x00, 0x01),
            HID_REPORT_SIZE(0x01),
            HID_REPORT_COUNT(0x01),
            HID_INPUT(0x02),

            // empty
            HID_REPORT_SIZE(0x07),
            HID_REPORT_COUNT(0x01),
            HID_INPUT(0x01),

            HID_USAGE_PAGE (0x01),

            // usage x
            HID_USAGE (0x30),
            HID_LOGICAL_MINMAX16(0x0000, 0x7fff),
            HID_REPORT_SIZE(0x10),
            HID_REPORT_COUNT(0x01),
            HID_INPUT(0x02),

            // usage y
            HID_USAGE (0x31),
            HID_LOGICAL_MINMAX16(0x0000, 0x7fff),
            HID_REPORT_SIZE(0x10),
            HID_REPORT_COUNT(0x01),
            HID_INPUT(0x02),
       HID_END_COLLECTION(),
    HID_END_COLLECTION()
};

// multi touch descriptor
static const uint8_t _multiTouchReportLen = 12;
static const uint8_t _multiTouch[] =
{
    HID_USAGE_PAGE (0x0D),
    HID_USAGE (0x04),
    HID_COLLECTION (0x01),
        // Digitizer
        HID_USAGE_PAGE (0x0D),
        HID_USAGE(0x22),
        HID_COLLECTION (0x02),
            HID_USAGE_PAGE (0x0D),
            HID_USAGE (0x38),

            HID_REPORT_SIZE(0x08),
            HID_REPORT_COUNT(0x01),
            HID_INPUT(0x02),

            HID_USAGE (0x33),
            HID_LOGICAL_MINMAX(0x00, 0x01),
            HID_REPORT_SIZE(0x01),
            HID_REPORT_COUNT(0x01),
            HID_INPUT(0x02),
            HID_REPORT_SIZE(0x07),
            HID_REPORT_COUNT(0x01),
            HID_INPUT(0x01),

            HID_USAGE_PAGE (0x01),
            HID_USAGE (0x30),
            HID_LOGICAL_MINMAX16(0x00, 0x7fff),

            HID_REPORT_SIZE(0x10),
            HID_REPORT_COUNT(0x01),

            HID_INPUT(0x02),
            HID_USAGE (0x31),

            HID_LOGICAL_MINMAX16(0x00, 0x7fff),
            HID_REPORT_SIZE(0x10),
            HID_REPORT_COUNT(0x01),
            HID_INPUT(0x02),
        HID_END_COLLECTION(),
        // Digitizer
        HID_USAGE_PAGE (0x0D),
        HID_USAGE(0x22),
        HID_COLLECTION (0x02),
            HID_USAGE_PAGE (0x0D),
            HID_USAGE (0x38),

            HID_REPORT_SIZE(0x08),
            HID_REPORT_COUNT(0x01),
            HID_INPUT(0x02),

            HID_USAGE (0x33),
            HID_LOGICAL_MINMAX(0x00, 0x01),
            HID_REPORT_SIZE(0x01),
            HID_REPORT_COUNT(0x01),
            HID_INPUT(0x02),
            HID_REPORT_SIZE(0x07),
            HID_REPORT_COUNT(0x01),
            HID_INPUT(0x01),

            HID_USAGE_PAGE (0x01),
            HID_USAGE (0x30),
            HID_LOGICAL_MINMAX16(0x00, 0x7fff),

            HID_REPORT_SIZE(0x10),
            HID_REPORT_COUNT(0x01),

            HID_INPUT(0x02),
            HID_USAGE (0x31),

            HID_LOGICAL_MINMAX16(0x00, 0x7fff),
            HID_REPORT_SIZE(0x10),
            HID_REPORT_COUNT(0x01),
            HID_INPUT(0x02),
        HID_END_COLLECTION(),
    HID_END_COLLECTION()
};

HIDDigitizer::HIDDigitizer(const string& inUUID, const string& inDisplayUUID, const string& inName,
        bool inMultiTouch, int inVendorId, int inProductId, int inCountryCode)
{
    // store strings
    uuid        = inUUID;
    displayUUID = inDisplayUUID;
    name        = inName;
    multiTouch  = inMultiTouch;

    // prepare device structure
    memset(&device, 0, sizeof(device));
    device.DisplayUUID = displayUUID.c_str();
    device.HIDCountryCode = inCountryCode;
    device.HIDProductId = inProductId;
    device.HIDVendorId = inVendorId;
    device.Name = name.c_str();
    device.UUID = uuid.c_str();

    if (multiTouch)
    {
        device.HIDDescriptor = _multiTouch;
        device.HIDDescriptorLen = sizeof(_multiTouch);
        // set report length
        report.HIDReportLen = _multiTouchReportLen;
    }
    else
    {
        device.HIDDescriptor = _singleTouch;
        device.HIDDescriptorLen = sizeof(_singleTouch);
        // set report length
        report.HIDReportLen = _singleTouchReportLen;
    }

    // prepare report template
    report.UUID = this->uuid.c_str();

    memset(data, 0, sizeof(data));
}

bool HIDDigitizer::Down(float inX, float inY, uint32_t inID, HIDInputReportWithData& outReport)
{
    if ((multiTouch && inID > 1) || (!multiTouch && inID > 0))
    {
        // ignore IDs above 2 fingers (or 1 finger for single touch)
        return false;
    }

    uint8_t id = inID & 0xff; // support only 8 bit indices

    uint16_t x = (uint16_t)(inX * (float)0x7fff);
    uint16_t y = (uint16_t)(inY * (float)0x7fff);

    uint8_t offset = 0;
    if (multiTouch)
    {
        data[0 + id * 6] = id;
        offset = (1 + id * 6);
    }
    data[0 + offset] = 1; // button down
    data[1 + offset] = (uint8_t)( x       & 0xff);
    data[2 + offset] = (uint8_t)((x >> 8) & 0xff);
    data[3 + offset] = (uint8_t)( y       & 0xff);
    data[4 + offset] = (uint8_t)((y >> 8) & 0xff);

    // copy report and data
    memcpy(outReport.data, data, report.HIDReportLen);
    memcpy(&outReport.report, &report, sizeof(report));
    outReport.report.HIDReportData = outReport.data;
    return true;
}

bool HIDDigitizer::Up(uint32_t inID, HIDInputReportWithData& outReport)
{
    if ((multiTouch && inID > 1) || (!multiTouch && inID > 0))
    {
        // ignore IDs above 2 fingers (or 1 finger for single touch)
        return false;
    }

    uint8_t id = inID & 0xff; // support only 8 bit indices
    uint8_t offset = 0;
    if (multiTouch)
    {
        data[0 + id * 6] = id;
        offset = (1 + id * 6);
    }
    data[0 + offset] = 0; // button up

    // copy report and data
    memcpy(outReport.data, data, report.HIDReportLen);
    memcpy(&outReport.report, &report, sizeof(report));
    outReport.report.HIDReportData = outReport.data;

    // reset data
    if (multiTouch)
        data[0 + id * 6] = id;
    data[0 + offset] = 0; // button up
    data[1 + offset] = 0;
    data[2 + offset] = 0;
    data[3 + offset] = 0;
    data[4 + offset] = 0;

    return true;
}

} } // namespace adit { namespace carplay
